% === postSimAnalysis.m ===
% Script para visualizar y exportar resultados de la simulación en Simulink.

% === 1. Extraer datos ===
Corriente       = squeeze(out.Corriente.Data);      % Corriente medida enviada por el microcontrolador
Tension         = squeeze(out.Tension.Data);        % Tensión aplicada al motor
Tiempo          = squeeze(out.TiempoReal.Data);     % Tiempo real capturado por ESP32
VelocidadReal   = squeeze(out.VelReal.Data);        % Velocidad real del encoder
VelocidadModelo = squeeze(out.VelModelo.Data);      % Velocidad estimada por el modelo matemático
Setpoint        = squeeze(out.Setpoint.Data);       % Referencia usada

% === 2. Crear figura principal ===
f = figure('Position', [10, 10, 1500, 700], 'Name', 'Análisis de Simulación');
p1 = uipanel('Parent', f, 'Units', 'normalized', 'Position', [0.05 0.05 0.8 0.9], 'Title', 'Gráfica');
ax = axes('Parent', p1);

% === 3. Dibujar curvas iniciales ===
yyaxis(ax, 'left')
h1 = plot(ax, Tiempo, VelocidadModelo, 'Color', [0 0 1], 'LineWidth', 1.5); hold on;
h2 = plot(ax, Tiempo, VelocidadReal, 'Color', [1 0 0],'LineStyle','-', 'LineWidth', 1.5);
h3 = plot(ax, Tiempo, Setpoint, 'Color', [0 0 0], 'LineStyle', '--', 'LineWidth', 1.5);
ylabel(ax, 'Velocidad [RPM]');
yyaxis(ax, 'right')
h4 = plot(ax, Tiempo, Tension, 'Color', [1 0 1], 'LineWidth', 1.5);
ylabel(ax, 'Tensión [V]');
xlabel(ax, 'Tiempo [s]');
legend([h1 h2 h3 h4], {'Velocidad Modelo','Velocidad Real','Setpoint','Tensión'}, 'Location', 'northwest');
title(ax, 'Velocidades, Setpoint y Tensión vs Tiempo');
grid(ax, 'on');

% === 4. Panel de controles ===
p3 = uipanel('Parent', f, 'Units', 'normalized', 'Position', [0.86 0.05 0.13 0.9], 'Title', 'Controles');
uicontrol('Parent', p3, 'Style', 'text', 'String', 'Color y visibilidad:', ...
          'Units', 'normalized', 'Position', [0.05 0.95 0.9 0.04], 'HorizontalAlignment', 'left', 'FontWeight', 'bold');

coloresMenu = {'Azul','Rojo','Verde','Negro','Magenta'};
dy = 0.07;
y0 = 0.92;

crearLineaControl(p3, 'Velocidad Modelo:', y0, coloresMenu, h1, 1); y0 = y0 - dy;
crearLineaControl(p3, 'Velocidad Real:', y0, coloresMenu, h2, 2); y0 = y0 - dy;
crearLineaControl(p3, 'Setpoint:', y0, coloresMenu, h3, 4); y0 = y0 - dy;
crearLineaControl(p3, 'Tensión:', y0, coloresMenu, h4, 5);

% === 5. Zoom ===
y0 = y0 - dy - 0.03;
uicontrol('Parent', p3, 'Style', 'text', 'String', 'Zoom:', ...
          'Units', 'normalized', 'Position', [0.1 y0 0.8 0.03], 'HorizontalAlignment', 'left', 'FontWeight', 'bold');
y0 = y0 - 0.04;
uicontrol('Parent', p3, 'Style', 'pushbutton', 'String', 'Zoom X+', ...
          'Units', 'normalized', 'Position', [0.1 y0 0.35 0.035], 'Callback', @(~,~) aplicarZoom(ax, 'x', 0.9));
uicontrol('Parent', p3, 'Style', 'pushbutton', 'String', 'Zoom X-', ...
          'Units', 'normalized', 'Position', [0.55 y0 0.35 0.035], 'Callback', @(~,~) aplicarZoom(ax, 'x', 1.1));
y0 = y0 - 0.045;
uicontrol('Parent', p3, 'Style', 'pushbutton', 'String', 'Zoom Y+', ...
          'Units', 'normalized', 'Position', [0.1 y0 0.35 0.035], 'Callback', @(~,~) aplicarZoom(ax, 'y', 0.9));
uicontrol('Parent', p3, 'Style', 'pushbutton', 'String', 'Zoom Y-', ...
          'Units', 'normalized', 'Position', [0.55 y0 0.35 0.035], 'Callback', @(~,~) aplicarZoom(ax, 'y', 1.1));

% === 6. Rango de tiempo ===
y0 = y0 - 0.06;
uicontrol('Parent', p3, 'Style', 'text', 'String', 'Tiempo [s]:', ...
          'Units', 'normalized', 'Position', [0.1 y0 0.8 0.03], 'HorizontalAlignment', 'left', 'FontWeight', 'bold');
y0 = y0 - 0.04;
uicontrol('Parent', p3, 'Style', 'text', 'String', 'Inicio:', ...
          'Units', 'normalized', 'Position', [0.1 y0 0.3 0.025], 'HorizontalAlignment', 'left');
editInicioX = uicontrol('Parent', p3, 'Style', 'edit', 'String', num2str(Tiempo(1)), ...
          'Units', 'normalized', 'Position', [0.4 y0 0.5 0.03]);
y0 = y0 - 0.045;
uicontrol('Parent', p3, 'Style', 'text', 'String', 'Fin:', ...
          'Units', 'normalized', 'Position', [0.1 y0 0.3 0.025], 'HorizontalAlignment', 'left');
editFinX = uicontrol('Parent', p3, 'Style', 'edit', 'String', num2str(Tiempo(end)), ...
          'Units', 'normalized', 'Position', [0.4 y0 0.5 0.03]);
y0 = y0 - 0.05;
uicontrol('Parent', p3, 'Style', 'pushbutton', 'String', 'Aplicar Rango', ...
          'Units', 'normalized', 'Position', [0.1 y0 0.8 0.04], 'Callback', @(~,~) aplicarRangoX(ax, editInicioX, editFinX));

% === 7. Exportar y guardar ===
y0 = y0 - 0.1;
uicontrol('Parent', p3, 'Style', 'pushbutton', 'String', 'Mostrar Tabla Corriente', ...
          'Units', 'normalized', 'Position', [0.1 y0 0.8 0.04], 'Callback', @(~,~) mostrarTablaCorriente(Tiempo, Corriente));
y0 = y0 - 0.06;
uicontrol('Parent', p3, 'Style', 'pushbutton', 'String', 'Exportar a Excel', ...
          'Units', 'normalized', 'Position', [0.1 y0 0.8 0.04], ...
          'Callback', @(~,~) exportarTablaExcel(Tiempo, VelocidadModelo, VelocidadReal, Setpoint, Tension, Corriente));
y0 = y0 - 0.06;
uicontrol('Parent', p3, 'Style', 'pushbutton', 'String', 'Guardar Gráfico', ...
          'Units', 'normalized', 'Position', [0.1 y0 0.8 0.04], 'Callback', @(~,~) guardarGrafico(f));

% === 8. Funciones auxiliares ===
function state = onoff(val)
    if val, state = 'on'; else, state = 'off'; end
end

function cambiarColor(h, val)
    colores = {[0 0 1], [1 0 0], [0 1 0], [0 0 0], [1 0 1]};
    h.Color = colores{val};
end

function crearLineaControl(parent, label, y, menu, handle, defaultIdx)
    uicontrol('Parent', parent, 'Style', 'text', 'String', label, ...
              'Units', 'normalized', 'Position', [0.1 y 0.8 0.025], 'HorizontalAlignment', 'left');
    uicontrol('Parent', parent, 'Style', 'popupmenu', 'String', menu, 'Value', defaultIdx, ...
              'Units', 'normalized', 'Position', [0.1 y-0.03 0.6 0.03], ...
              'Callback', @(src,~) cambiarColor(handle, src.Value));
    uicontrol('Parent', parent, 'Style', 'checkbox', 'Value',1, 'String','', ...
              'Units', 'normalized', 'Position', [0.75 y-0.03 0.2 0.03], ...
              'Callback', @(src,~) set(handle, 'Visible', onoff(src.Value)));
end

function aplicarZoom(axHandle, eje, factor)
    if eje=="x"
        xl = xlim(axHandle);
        xmid = mean(xl);
        xrange = diff(xl)*factor/2;
        xlim(axHandle, [xmid - xrange, xmid + xrange]);
    elseif eje=="y"
        yl = ylim(axHandle);
        ymid = mean(yl);
        yrange = diff(yl)*factor/2;
        ylim(axHandle, [ymid - yrange, ymid + yrange]);
    end
end

function aplicarRangoX(axHandle, editInicioX, editFinX)
    inicio = str2double(editInicioX.String);
    fin = str2double(editFinX.String);
    if isnan(inicio) || isnan(fin) || inicio >= fin
        errordlg('Inicio < Fin y deben ser numéricos','Error');
        return;
    end
    xlim(axHandle, [inicio fin]);
end

function mostrarTablaCorriente(Tiempo, Corriente)
    f2 = figure('Name', 'Tabla de Corriente', 'Position', [500, 100, 400, 600]);
    uitable('Parent', f2, 'Data', [Tiempo Corriente], ...
            'ColumnName', {'Tiempo [s]', 'Corriente [mA]'}, ...
            'Units', 'normalized', 'Position', [0.05 0.2 0.9 0.75]);
end

function exportarTablaExcel(Tiempo, VelModelo, VelReal, Setpoint, Tension, Corriente)
    T = table(Tiempo, VelModelo, VelReal, Setpoint, Tension, Corriente);
    filename = 'ResultadosSimulacion.xlsx';
    writetable(T, filename);
    msgbox(['Tabla exportada: ', filename]);
end

function guardarGrafico(figHandle)
    [file,path] = uiputfile({'*.png';'*.fig'}, 'Guardar gráfico');
    if isequal(file,0), return; end
    [~,~,ext] = fileparts(file);
    if strcmp(ext,'.fig')
        savefig(figHandle, fullfile(path,file));
    else
        saveas(figHandle, fullfile(path,file));
    end
    msgbox(['Guardado en: ', fullfile(path,file)]);
end
